import os
import logging
from flask import Flask, render_template, request, flash, redirect, url_for
from flask_wtf.csrf import CSRFProtect
from forms import ContactForm

# Configure logging
logging.basicConfig(level=logging.DEBUG)

# Create the Flask app
app = Flask(__name__)
app.secret_key = os.environ.get("SESSION_SECRET", "dev-secret-key-change-in-production")

# Initialize CSRF protection
csrf = CSRFProtect(app)

@app.route('/')
def index():
    """Homepage route"""
    return render_template('index.html')

@app.route('/about')
def about():
    """About us page route"""
    return render_template('about.html')

@app.route('/services')
def services():
    """Services page route"""
    return render_template('services.html')

@app.route('/contact', methods=['GET', 'POST'])
def contact():
    """Contact page route with form handling"""
    form = ContactForm()
    
    if form.validate_on_submit():
        # In a real application, you would save this to a database or send an email
        name = form.name.data
        email = form.email.data
        phone = form.phone.data
        message = form.message.data
        
        # Log the contact form submission
        app.logger.info(f"Contact form submitted by {name} ({email})")
        
        flash('Thank you for your message! We will get back to you soon.', 'success')
        return redirect(url_for('contact'))
    
    return render_template('contact.html', form=form)

@app.errorhandler(404)
def page_not_found(e):
    """404 error handler"""
    return render_template('base.html', 
                         page_title="Page Not Found", 
                         error_message="The page you're looking for doesn't exist."), 404

@app.errorhandler(500)
def internal_server_error(e):
    """500 error handler"""
    return render_template('base.html', 
                         page_title="Server Error", 
                         error_message="Something went wrong on our end. Please try again later."), 500

if __name__ == '__main__':
    app.run(debug=True, host='0.0.0.0', port=5000)
